/*
    autor: Marcin Jdrzejewski (marcinj at aster dot pl)
    opis: plik efektw do algorytmu rzucania promieni na GPU
    data: 2004-08-01
*/
static float EPSILON = 0.001f;
static float bigNumber=9999.0f;

//maksymalna odlego na ktr propagowane s promienie
// okrelenie maksymalnej dugoci echogramu
const float maxDistance; 

//sfera odbiornika moe znajdowa si w wielu liciach drzewa
// BSP na raz0, tu maksymalnie moe znajdowa si w 4-rech
const float receiversLeaf;
const float receiversLeaf2;
const float receiversLeaf3;
const float receiversLeaf4;

//rednica sfery podniesiona do kwadratu
const float receiverRaySqRadius;
//pozycja sfery odbiornika
const float3 listenerPos;

//tekstury
//paszczyzny (ciany i portale)
texture PlaneData;
//parametry paszczyzn
texture PlaneParamData;
//informacje o liciach drzewa BSP
texture LeafData;

//render target textures
texture Rt1Data; //stany promieni
texture Rt2Data; 
texture Rt3Data;

//macierze transformacji
float4x4 World      : WORLD;
float4x4 View       : VIEW;
float4x4 Projection : PROJECTION;

//wyjcie vertex shadera
struct VS_OUTPUT
{
    float4 Pos  : POSITION;    
    float2 Tex : TEXCOORD0;    
};

//wyjcie piksel shadera
struct PS_OUTPUT
{
    float4 state1 : COLOR0;            
    float4 data1  : COLOR1;   
    float4 data2  : COLOR2;    
};

//standardowy vertex shader
VS_OUTPUT VS(
    float3 Pos  : POSITION, 
    float3 Norm : NORMAL, 
    float2 Tex  : TEXCOORD0)
{
    VS_OUTPUT Out = (VS_OUTPUT)0;    
    float4x4 WorldView = mul(World, View);        
    float3 P = mul(Pos, WorldView);    
    Out.Pos  = mul(float4(P, 1), Projection);
    Out.Tex = Tex;
    return Out;
}

//kada tekstur musi by skojarzona z samplerem i dopiero
// przez niego mona pobiera dane w pixel shaderze
// filtrowanie jest wyczona 
sampler SamplerPlaneData = sampler_state
{
    Texture   = (PlaneData);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};
sampler SamplerPlnParams = sampler_state
{
    Texture   = (PlaneParamData);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};
sampler SamplerLeafData = sampler_state
{
    Texture   = (LeafData);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};

sampler StateData1 = sampler_state
{
    Texture   = (Rt1Data);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};
sampler StateData2 = sampler_state
{
    Texture   = (Rt2Data);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};
sampler StateData3 = sampler_state
{
    Texture   = (Rt3Data);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};


// oznaczenia zawartoci poszczeglnych pl tekstur stanu
//
//state1.x   <- aktualny li drzewa BSP w ktrym znajduje si ten 
//               promie
//state1.y   <- aktualna paszczyzna rozpatrywana w aktualnym liciu, 
//               indeksowana
//               od 0 do maksymalnej liczby paszczyzn dla tego licia  
//state1.z   <- wspczynnik absorpcji obliczony poprzez mnoenie 
//               wspczynnikw absorpcji kolejnych scian z ktrymi
//               doszo do kolizji 
//state1.w   <- odlego do paszczyzny najbliszej pocztkowi 
//               promienia
//data1.xyz  <- punkt w ktrym promie ma swj pocztek
//data1.w    <- cakowita odlego jak przeby promie, warto ta 
//              jest ujemna jeli doszo do przecicia ze sfer
//              (optymalizacja kodu)
//data2.xyz  <- kierunek promienia
//data2.w    <- indeks paszczyzny ktra znajduje si najbliej 
//               punktu pocztku promienia
float4 PS_LeafPlaneIntersect(    
    float2 Tex : TEXCOORD0
   ) : COLOR
{   
    float4 psout;
    psout = tex2D(StateData1 , Tex);
        
    float4 data1 = tex2D(StateData2 , Tex);
    float4 data2 = tex2D(StateData3 , Tex);
    half pos = data2.w*1.0f/2048.0f + 0.5f/2048.0f;    
    half dist;
            
    for (  half i=-1; i>=-6; i--)
    {    
        float4 plane = tex1D(SamplerPlaneData, pos);
                        
        dist = plane.w - dot(plane.xyz, data1.xyz);
        half dotres = dot (plane.xyz, data2.xyz);
        dist /= dotres;
        half outp=99999;
                
        if ( dotres<0 ) outp = dist;        
        if ( outp < psout.x ) psout.y = i;
        if ( outp < psout.x ) psout.x = outp;
            
        pos += 1.0f/2048.0f;
    }

    if ( psout.y < 0)
        psout.y=data2.w+(-psout.y-1);
    
    return psout;
}

PS_OUTPUT PS_Propagate(    
    float2 Tex : TEXCOORD0
   )
{   
    PS_OUTPUT psout;   
    
    psout.state1 = tex2D(StateData1 , Tex);        
    psout.data1 = tex2D(StateData2 , Tex);
    psout.data2 = tex2D(StateData3 , Tex);

    half pos;
    float4 plane;
    float4 params;
    
    //all planes for this leaf were checked so, use index of the one stored    
    pos = psout.state1.y*1.0f/2048.0f;
    
    float4 leaf_data = 
        tex1D(SamplerLeafData , 
              psout.data1.w*1.0f/2048.0f + 0.5f/2048.0f);      

    // plane.xyz - plane normal, plane.w distance from origin    
    plane = tex1D(SamplerPlaneData, pos);
    
    //params.x <- equals 1 if its a plane, 0 if its a portal
    //params.y <- equals leaf number that should be set for next iteration,
    //             if its plane then this ray will stay in the same leaf
    //             if it was portal then this value will be leaf number of 
    //               neighbouring leaf where ray should now travel
    //params.z <- equals attenuation coeficient of surface
    params = tex1D(SamplerPlnParams, pos); 
    
    if ( psout.state1.z < 0 )
        psout.data2.w = -100;    
    
    if ( psout.data2.w+6 < leaf_data.y )
    {        
        psout.data2.w += 6;
    }
    else
    {
        //now we have found closest plane/portal
        
        //first check if receiver have been 
        // struckt with this ray (just test for it)
        half s=0;               
        
        half test = psout.data1.w - receiversLeaf;
        test *= psout.data1.w - receiversLeaf2;
        test *= psout.data1.w - receiversLeaf3;
        test *= psout.data1.w - receiversLeaf4;
        
        if ( test == 0)   
        {   
            // more on this code in Real-Time rendering page.571
            
            // vector from ray origin to the sphere position
            float3 L = listenerPos - psout.data1.xyz;
            // distance to the plane containing center of sphere
            s = dot(L, psout.data2.xyz);             
            half S2 = s*s;            
            // squared distance to center of sphere
            half L2 = dot(L,L);                      
            // no intersection if ray points away from sphere
            //   and is not inside it
            if ( s < 0 && L2 > receiverRaySqRadius)  
                s = 0;                               
            half m2 = L2 - S2;// from Pythagorean theorem
            if ( m2 > receiverRaySqRadius )          
                s = 0;                               
            
            
            float q = sqrt(receiverRaySqRadius - m2);
            if ( s != 0 )
            {
            if ( L2 > receiverRaySqRadius)
                s = s - q;
            else
                s = s + q;
            } 
                                       
        }
                        
        if ( s > 0 )
        {
            psout.state1.x = s; //distance to closest plane     
            psout.state1.z*=-1;       
            params.z = 1;
            params.x = 0;
        }
        
        //we have tested all planes for this leaf so, use the closest
        // as intersecting one
        //leaf number - use the one from plane data
        psout.data1.w = params.y;   
        
        float4 new_leaf_data = tex1D(SamplerLeafData,
            params.y*1.0f/2048.0f + 0.5f/2048.0f);      

        //set tested plane count to zero                        
        psout.data2.w = (new_leaf_data.x - 0.5f/2048.0f) * 2048.0f;

        //add plane attenuation cooeficient (one for portals)                
        psout.state1.z *= params.z;     
        
        //compute intersection point
        psout.data1.xyz += psout.data2.xyz*psout.state1.x;    
        
        //current distance this ray have traveled 
        psout.state1.w+= psout.state1.x;

        //set distance to closest plane to some big number                                                
        psout.state1.x = bigNumber;

        //if it is a plane, then reflect
        if ( params.x )
        {
            //reflect ray direction
            psout.data2.xyz = reflect(psout.data2.xyz, plane.xyz); 
        }
        
        //if receiver sphere hit - then stop ray propagation            
        //if reached max reflections then stop propagataion
        //if ray is too weak then also stop propagation
                
        /*
        if (s>=0)
            psout.state1.z+=1;        
        if (psout.state1.w > maxDistance)
        {        
            psout.state1.w*=-1;
        }
        */
        
    }       
    return psout;

}

//do debugowania, tworzy teksture umoliwiajc wizualn
// ocen postpw algorytmu
float4 PS_MakeOnScreenTexture(    
    float2 Tex : TEXCOORD0
   ) : COLOR
{   
   PS_OUTPUT ps_in;
   float4 out_col=0;
    
   ps_in.state1 = tex2D(StateData1 , Tex);    
   ps_in.data1 = tex2D(StateData2 , Tex);
   ps_in.data2 = tex2D(StateData3 , Tex);

    float refs = log(abs(ps_in.state1.z))/log(0.7);        

    if ( ps_in.state1.z < 0 )
    {
        if ( refs == 0 )
            out_col.r = 1;
        else
        if ( abs(refs - 1)< 0.1f)
            out_col.g = 1;
        else    
        if ( abs(refs - 2)< 0.1f)
            out_col.b = 1;
        else
        //if ( refs > 2 )
            out_col = 1.0f-(refs*10.0f)/255.0f;
    }
            
    return out_col;

}

//piksel shader kopiujcy wynik oblicze do jednej tekstury
// wyjciowej 
float4 PS_PackTo32BitTexture (
    float2 Tex : TEXCOORD0
   ) : COLOR
{   
    float4 psout=0;
    float4 state1 = tex2D(StateData1 , Tex);

    psout.x = 1.0f;    
    psout.x *= abs(state1.z);
    psout.x += floor(state1.w);
    psout.x *= sign(state1.z);

    return psout;
}

//dla kadego promienia oblicza pozycje jego wirtualnego rda
// dwikowego
float4 PS_ComputeImageSources (
    float2 Tex : TEXCOORD0
   ) : COLOR
{   
   PS_OUTPUT ps_in;   
   float4 out_col=0;
    
   ps_in.state1 = tex2D(StateData1 , Tex);    
      
   //punkt pocztkowy promienia
   ps_in.data1 = tex2D(StateData2 , Tex); 
   //kierunek promienia
   ps_in.data2 = tex2D(StateData3 , Tex); 
   
   //obliczanie pozycji rda dwiku
   ps_in.data2 *= -1.0f;
   ps_in.data1 += ps_in.state1.w * ps_in.data2;    
   out_col.xyz = ps_in.data1.xyz; //pozycja
   out_col.w = ps_in.state1.z;   //absorpcja
            
   return out_col;
}

//techniki uruchamiane z poziomu aplikacji
technique LeafPlaneIntersect
{
    pass P0
    {
        // shaders
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS_LeafPlaneIntersect();
    }  
}

technique ATIRadeon9800
{
   pass P0 //przecicia z paszczyznami
    {        
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS_LeafPlaneIntersect();        
    }  
   pass P1 //propagacja 
    {        
        VertexShader = compile vs_2_0 VS();        
        PixelShader  = compile ps_2_0 PS_Propagate();        
    }  
    pass P2 //pakowanie do tekstury wyjciowej
    {        
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS_ComputeImageSources();
    }          
}

technique MakeOnScreenTexture
{
  pass P0 //pokazuje stany promieni na ekranie
    {     
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS_MakeOnScreenTexture();
    }  
}

